/*
 *  auth.js
 *
 *  Copyright: (c) 2018 FileMaker, Inc. All rights reserved.
 *
 *  FILEMAKER CONFIDENTIAL
 *  This file includes unpublished proprietary source code of FileMaker,
 *  Inc. The copyright notice above does not evidence any actual or
 *  intended publication of such source code.
 */
'use strict';
var util = require('../routes/util');
var errorHandler = require('../routes/error');

module.exports.login = function(req, res, next){
	if(!req.is('application/json')){
 		return errorHandler.handleError("UnsupportedMediaType", req, res);
	}
	var missingParam = [];
	var ipStr = util.parseClientIp(req);
	var params = {
		'solution': req.swagger.params.database.value,
		'requestMethod': req.method,
		'requestPath': req.originalUrl,
		'requestIp': ipStr,
		'requestHostname' : 'https://' + req.get("X-Forwarded-Host"), // get X-Forwarded-Host instead of req.hostname to get port number as well and prepend https
		'inputBandwidth': (req.body) ? JSON.stringify(req.body).length.toString() : "0",
		'version': req.swagger.params.version.value
	};

	var result = util.setAuthentication(req, res, params, true);	
	if(result) { //continue only if setAuthentication returns everything is OK
		util.parseParams(['fmDataSource'], req.body, params);

		var remainingProps = Object.keys(req.body);
		if (remainingProps.length > 0) {
			return errorHandler.handleError("BadRequest", req, res, "Unknown parameter(s): "+remainingProps.join(','),"960");		
		}

		if (params['fmDataSource']) {
			var fmds = (typeof params['fmDataSource'] === "string") ? JSON.parse(params['fmDataSource']) : params['fmDataSource'];
			if (!Array.isArray(fmds)) {
				return errorHandler.handleError('BadRequest', req, res, "fmDataSource parameter must be an array","960");			
			}
			for (var i = 0; i < fmds.length; i++) {
				var entry = fmds[i];
				if (typeof entry !== "object") {
					return errorHandler.handleError('BadRequest', req, res, "fmDataSource values must be an object","960");			
				}
				if ((entry.hasOwnProperty('username') && !entry.hasOwnProperty('password')) ||
					(!entry.hasOwnProperty('username') && entry.hasOwnProperty('password'))) {
					return errorHandler.handleError("BadRequest", req, res, "fmDataSource FM credentials requires both 'username' and 'password' properties","10");		
				}
				if ((entry.hasOwnProperty('oAuthRequestId') && !entry.hasOwnProperty('oAuthIdentifier')) ||
					(!entry.hasOwnProperty('oAuthRequestId') && entry.hasOwnProperty('oAuthIdentifier'))) {
					return errorHandler.handleError("BadRequest", req, res, "fmDataSource OAuth requires both 'oAuthRequestId' and 'oAuthIdentifier' properties","10");		
				}
				for (var propName in entry) {
					if (['database','username','password','oAuthRequestId','oAuthIdentifier'].indexOf(propName) === -1) {
						return errorHandler.handleError("BadRequest", req, res, "Unknown fmDataSource parameter: '"+propName+"'","960");		
					}
				}
			}
			if (typeof params['fmDataSource'] !== "string") {
				params['fmDataSource'] = JSON.stringify(params['fmDataSource']);
			}

		}

		if(missingParam.length > 0){
			//per Bug 208594, add errorcode "958"
			return errorHandler.handleError("BadRequest", req, res, "Missing parameter(s): "+missingParam.join(','),"958");
		}

		try {
			util.thrift_client.login(
				params,
				function(thrifError, thrifResult) {
					return util.handleThrifReturn(thrifError, thrifResult, req, res, function(thrifResult, res) {
						var result = JSON.parse(thrifResult);
						if (result.response['token']) {
							var token = result.response['token'];
							res.set('X-FM-Data-Access-Token', token);
						}
					});
				});
		} catch (err) {
			util.thriftExceptionHandler(err, req, res);
		}
	} //end of if
}

module.exports.logout = function(req, res, next) {
	if (util.validateRequest(req, res) === false) {
		return;
	}
	try {
		
		var ipStr = util.parseClientIp(req);
		var params = {
			'solution': req.swagger.params.database.value,
			'requestMethod': req.method,
			'requestPath': req.originalUrl,
			'requestIp': ipStr,
			'version': req.swagger.params.version.value,
			'token': req.swagger.params.sessionToken.value
		};
		util.thrift_client.logout(
			params,
			function(thrifError, thrifResult) {
				return util.handleThrifReturn(thrifError, thrifResult, req, res);
			}
		);
	} catch (err) {
		util.thriftExceptionHandler(err, req, res);
	}
}

module.exports.validateSession = function(req, res, next) {
	if (util.validateRequest(req, res) === false) {
		return;
	}
	try {
		var params = {
			'requestMethod': req.method,
			'requestPath': req.originalUrl,
			'requestIp': req.ip,
			'version': req.swagger.params.version.value
		};
		var result = util.setAuthentication(req, res, params, false);	
		if(result){
			util.thrift_client.validateSession(
				params,
				function(thrifError, thrifResult) {
					return util.handleThrifReturn(thrifError, thrifResult, req, res);
				}
			);
		}
	} catch (err) {
		util.thriftExceptionHandler(err, req, res);
	}
}

